

/***************************************************
Title: O3Q Light Sensor Module
Description: This is an Arduino code for the O3Q light sensor module, which uses
an ESP32 D1 Mini NodeMCU board with a BH1750 ambient light sensor. The code creates
a WiFi access point that allows users to connect to the sensor via a web interface. 
The web interface can be accessed at <http://o3q.local/> or <http://192.168.4.1>
Author: Nils Haverkamp, Institut für Didaktik der Physik, Universität Münster
Date: February 20, 2025
Hardware: ESP32 D1 Mini NodeMCU, BH1750 Ambient Light Sensor
Software: Arduino IDE 1.8.19
SSID: <O3Q-Lichtsensor *>
Key: <O3Q.de_LS*>
(The asterisk represents the serial number that is set in line 15 of the code.)
********************************************/
byte SerialNumber=66;

#include <WiFi.h>
#include <AsyncTCP.h>
#include <ESPAsyncWebServer.h>
//#include <SPIFFS.h>
#include <ESPmDNS.h>
#include <Wire.h>
#include "BH1750_WE.h"
#include "Website.h"

/******************Define************************/

//Wifi- AP
AsyncWebServer server(80); 
BH1750_WE BH1750(0x23);// Fläche der Photodiode: 0,25mm x 0,3mm; 1lx=1lm/m^2 1W==683lm(Photometrisches Strahlungsäquivalent)

String ssid = "O3Q-Lichtsensor ", password = "O3Q.de_LS";
String ownSSID;
char* hostname="O3Q";

//Loop & Measurements
bool BigMeasuringRange = 0;
float lightIntensity;



/******************Setup************************/


void setup(){
  Serial.begin(115200);
  pinMode(5, OUTPUT)  ,pinMode(18, OUTPUT)  ,pinMode(26, OUTPUT), pinMode(2, OUTPUT);
  digitalWrite(5,LOW) ,digitalWrite(18,LOW) ,digitalWrite(26,HIGH),digitalWrite(2, HIGH);
  delay(100);
  
  Wire.begin(23,19);
  BH1750.init(); 
  BH1750.setMeasuringTimeFactor(1.67);
  //SPIFFS.begin();
  setupWifi();
    
  // Handle HTTP 
  HandleHTTPRequests(); 
  server.begin();
  digitalWrite(2, LOW);//Onbord LED
}


void loop(){
  //constantly refresh light measurement of BH1750
  lightIntensity = BH1750.getLux();
  if (lightIntensity>10000&&!BigMeasuringRange){
    BigMeasuringRange = 1;
    BH1750.setMeasuringTimeFactor(0.45);
  }
  if(lightIntensity<10000&&BigMeasuringRange){
    BigMeasuringRange = 0;
    BH1750.setMeasuringTimeFactor(1.67);
  }
}

void setupWifi(){
  WiFi.mode(WIFI_AP_STA);
  ownSSID=ssid+SerialNumber;
  int channel=11, ssid_hidden=0, max_connection=2;
  WiFi.softAP(ownSSID, password+SerialNumber, channel, ssid_hidden, max_connection);
  Serial.println(ownSSID);
  IPAddress IP = WiFi.softAPIP();
  Serial.print("IP address: ");
  Serial.println(IP);  
  Serial.print("Connect to webpage via: http://");
  Serial.print(hostname);
  Serial.println(".local");
  if (!MDNS.begin(hostname)) {  
    Serial.println("Error setting up MDNS responder!");
    while(1) {delay(1000);}
  }
}

void HandleHTTPRequests(){
  server.on("/"             , HTTP_GET, [](AsyncWebServerRequest *request){request->send(200, "text/html",  index_html);});  
  server.on("/Graph_I.html" , HTTP_GET, [](AsyncWebServerRequest *request){request->send(200, "text/html",Graph_I_html);});
  server.on("/infos.html"   , HTTP_GET, [](AsyncWebServerRequest *request){request->send(200, "text/html",  infos_html);});
  server.on("/PlotStuff.js" , HTTP_GET, [](AsyncWebServerRequest *request){request->send(200, "text/js",  PlotStuff_js);});
  server.on("/Graph_I.js"   , HTTP_GET, [](AsyncWebServerRequest *request){request->send(200, "text/js",    Graph_I_js);});
  server.on("/style.css"    , HTTP_GET, [](AsyncWebServerRequest *request){request->send(200, "text/css",    style_css);});
  server.on("/index.html"   , HTTP_GET, [](AsyncWebServerRequest *request){request->redirect("/")                      ;});
  server.onNotFound(                    [](AsyncWebServerRequest *request){request->redirect("/")                      ;});
  server.on("/measure"      , HTTP_GET, [](AsyncWebServerRequest *request){request->send_P(200, "text/plain", String(lightIntensity).c_str());});
}
